

module Examples where

open import Library
open import Types
open import Terms
open import Substitutions

open import Evaluation

-------------------------
--- Types

-- Guarded recursive streams of natural numbers

gStr : ∀{n} {Θ : TyCxt n} → Ty Θ
gStr {n} {Θ} = μₜ ℕₜ ×ₜ ▸ₜ (Varₜ α)
  where
    α : TyVar (¬⛨ ∷ Θ)
    α = zero

-- Coinductive streams

Str : ∀{n} {Θ : TyCxt n} → Ty Θ
Str = ■ₜ gStr


---------------------------
-- Fixed point combinators

-- We define a version of the Y combinator, which typechecks in gλ.

-- Y ≜ λf.δδ
-- δ ≜ λx. f((λz.unfold z) ⊛ x ⊛ next x) : ▸B → A
-- B ≜ μα. ▸α → A

-- Unfortunately, these need to be defined individually for each A at the
-- moment.

-- fixpoint operator for gStrₜ

fix-gStr : ∀{Γ} → Tm Γ ((▸ gStr →ₜ gStr) →ₜ gStr)
fix-gStr = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = gStr
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for ℕₜ →ₜ gStr (iterate)

iterateTy : ∀{n} {Θ : TyCxt n} → Ty Θ
iterateTy = ℕₜ →ₜ gStr

fix-iterate : ∀{Γ} → Tm Γ ((▸ iterateTy →ₜ iterateTy) →ₜ iterateTy)
fix-iterate = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = iterateTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for ℕₜ →ₜ ℕₜ →ₜ gStr (fib)

fibTy : ∀{n} {Θ : TyCxt n} → Ty Θ
fibTy = ℕₜ →ₜ ℕₜ →ₜ gStr

fix-fib : ∀{Γ} → Tm Γ ((▸ fibTy →ₜ fibTy) →ₜ fibTy)
fix-fib = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = fibTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for gStr →ₜ ▸ gStr →ₜ gStr (interleave)

interleaveTy : ∀{n} {Θ : TyCxt n} → Ty Θ
interleaveTy {Θ = Θ} = gStr →ₜ ▸ₜ_ {Θ′ = Θ} gStr →ₜ gStr

fix-interleave : ∀{Γ} → Tm Γ ((▸ interleaveTy →ₜ interleaveTy) →ₜ interleaveTy)
fix-interleave = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = interleaveTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for Str →ₜ gStr (every2nd)

every2ndTy : ∀{n} {Θ : TyCxt n} → Ty Θ
every2ndTy = Str →ₜ gStr

fix-every2nd : ∀{Γ} → Tm Γ ((▸ every2ndTy →ₜ every2ndTy) →ₜ every2ndTy)
fix-every2nd = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = every2ndTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for (ℕₜ →ₜ ℕₜ →ₜ ℕₜ) →ₜ gStr →ₜ gStr →ₜ gStr (zipWith)

zipWithTy : ∀{n} {Θ : TyCxt n} → Ty Θ
zipWithTy = (ℕₜ →ₜ ℕₜ →ₜ ℕₜ) →ₜ gStr →ₜ gStr →ₜ gStr

fix-zipWith : ∀{Γ} → Tm Γ ((▸ zipWithTy →ₜ zipWithTy) →ₜ zipWithTy)
fix-zipWith = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = zipWithTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-- fixpoint operator for .. (foldr-gStr)

foldr-gStrTy : ∀{n} {Θ : TyCxt n} → Ty Θ
foldr-gStrTy {Θ = Θ} = ((ℕₜ ×ₜ ▸ₜ_ {Θ′ = Θ} gStr) →ₜ gStr) →ₜ gStr →ₜ gStr

fix-foldr-gStr : ∀{Γ} → Tm Γ ((▸ foldr-gStrTy →ₜ foldr-gStrTy) →ₜ foldr-gStrTy)
fix-foldr-gStr = ƛ δ ∙ (next fold δ)
  where
    A : ∀{n} {Θ : TyCxt n} → Ty Θ
    A = foldr-gStrTy
    B : ∀{n} {Θ : TyCxt n} → Ty Θ
    B {n} {Θ} = μₜ ▸ₜ Varₜ {Θ = ¬⛨ ∷ Θ} zero →ₜ A
    nxtunfld : ∀{Γ} → Tm (▸ₜ B ∷ (▸ A →ₜ A) ∷ Γ) (▸ₜ (B →ₜ ▸ₜ B →ₜ A))
    nxtunfld = next (ƛ unfold (var zero))
    δ : ∀{Γ} → Tm ((▸ A →ₜ A) ∷ Γ) (▸ₜ B →ₜ A)
    δ = ƛ var (suc zero) ∙ nxtunfld ⊛ var zero ⊛ next var zero

-------------------------
-- Stream operations

-- Taking the head and tail of a gstream:

ghd : ∀{Γ} → Tm Γ (gStr →ₜ ℕₜ)
ghd = ƛ (π₁ (unfold (var zero)))

gtl : ∀{Γ} → Tm Γ (gStr →ₜ ▸ gStr)
gtl = ƛ π₂ (unfold (var zero))

-- Head and tail of coinductive streams

hd : ∀{Γ} → Tm Γ (Str →ₜ ℕₜ)
hd = ƛ ghd ∙ unbox (var zero)

tl : ∀{Γ} → Tm Γ (Str →ₜ Str)
tl = ƛ box prev gtl ∙ unbox (var zero) wth var zero wth var zero

-- A constructor for gstreams:

cons : ∀{Γ} → Tm Γ (ℕₜ →ₜ ▸ gStr →ₜ gStr)
cons = ƛ ƛ fold ⟨ var (suc zero) , var zero ⟩

-- Projections on (coinductive) streams.

nth : ∀{Γ} → ℕ → Tm Γ (Str →ₜ ℕₜ)
nth zero = hd
nth (suc n) = ƛ (nth n) ∙ (tl ∙ var zero)

-- A function that interleaves two streams:

-- interleave s₁ s₂ = head s₁ ∷ interleave s₂ (gtl s₁)

interleave : ∀{Γ} → Tm Γ (gStr →ₜ ▸ gStr →ₜ gStr)
interleave = fix-interleave ∙ (ƛ
              (ƛ ƛ cons ∙ (ghd ∙ var (suc zero)) ∙ (var (suc (suc zero)) ⊛ var zero ⊛ next (gtl ∙ var (suc zero)))))

-- interleave′ s₁ s₂ = head s₁ ∷ head s₂ ∷ interleave (gtl s₁) (gtl s₂)

-- interleave′ : ∀{Γ} → Tm Γ (gStr →ₜ ▸ gStr →ₜ gStr)
-- interleave′ = fix-interleave ∙ (ƛ (ƛ ƛ cons ∙ ((ghd ∙ var (suc zero))) ∙ {!!}))

interleave′ : ∀{Γ} → Tm Γ (gStr →ₜ ▸ gStr →ₜ gStr)
interleave′ = fix-interleave ∙ interleave′′
  where
  interleave′′ : ∀{Γ} → Tm Γ (▸ (gStr →ₜ ▸ₜ gStr →ₜ gStr) →ₜ gStr →ₜ ▸ gStr →ₜ gStr)
  interleave′′ = ƛ ƛ ƛ cons ∙ (ghd ∙ var (suc zero)) ∙
                           next cons ⊛ (next ghd ⊛ var zero) ⊛
                             next (var (suc (suc zero)) ⊛
                                       (gtl ∙ var (suc zero)) ⊛ (next gtl ⊛ var zero))

-- A higher-order function that zips two streams with a function:

-- zipWith f s₁ s₂ = f (ghd s₁) (ghd s₂) ∷ zipWith f (gtl s₁) (gtl s₂)

zipWith : ∀{Γ} → Tm Γ ((ℕₜ →ₜ ℕₜ →ₜ ℕₜ) →ₜ gStr →ₜ gStr →ₜ gStr)
zipWith = fix-zipWith ∙ (ƛ
            (ƛ ƛ ƛ cons ∙ (var (suc (suc zero)) ∙ (ghd ∙ var (suc zero)) ∙ (ghd ∙ var zero)) ∙
              var (suc (suc (suc zero))) ⊛ next var (suc (suc zero)) ⊛ (gtl ∙ var (suc zero)) ⊛ (gtl ∙ var zero)))

-- Adding two streams together:

-- plus = zipWith _+_

plus : ∀{Γ} → Tm Γ (gStr →ₜ gStr →ₜ gStr)
plus = zipWith ∙ (ƛ ƛ var zero ⊹ var (suc zero))

-------------------------
-- Concrete streams

-- A stream of zeros, defined in the obvious way:

-- zeros = 0 ∷ zeros

zeros : ∀{Γ} → Tm Γ Str
zeros = box
        fix-gStr ∙ (ƛ fold ⟨ zero , var zero ⟩)
        wth ⟨⟩

-- The same stream of zeros, defined in a less obvious way using a stream
-- function (this term cannot be defined in Agda without sized types):

-- zeros-ho = 0 ∷ interleave zeros-ho (gtl zeros-ho)

zeros-ho : ∀{Γ} → Tm Γ gStr
zeros-ho = fix-gStr ∙ zeros′
  where
    zeros′ : ∀{Γ} → Tm Γ (▸ gStr →ₜ gStr)
    zeros′ = ƛ cons ∙ zero ∙ (next interleave ⊛ var zero ⊛ (next gtl ⊛ var zero))

zeros-ho′ : ∀{Γ} → Tm Γ Str
zeros-ho′ = box fix-gStr ∙ zeros′ wth ⟨⟩
  where
    zeros′ : ∀{Γ} → Tm Γ (▸ gStr →ₜ gStr)
    zeros′ = ƛ cons ∙ zero ∙ (next interleave′ ⊛ var zero ⊛ (next gtl ⊛ var zero))


-- Generating a stream by iterating a function

-- iterate f (x ∷ xs) = f x ∷ iterate f xs

iterate : ∀{Γ} → Tm Γ ((ℕₜ →ₜ ℕₜ) →ₜ ℕₜ →ₜ gStr)
iterate = ƛ fix-iterate ∙ (ƛ ƛ cons ∙ var zero ∙ var (suc zero) ⊛ next (var (suc (suc zero)) ∙ var zero))

-- A stream of natural numbers, starting in zero:

-- nats = iterate suc 0

nats : ∀{Γ} → Tm Γ Str
nats = box iterate ∙ (ƛ suc (var zero)) ∙ zero wth ⟨⟩

-- A stream containg the Fibonacci sequence, 0, 1, 1, 2, ...:

-- fib′ n m = n ∷ m ∷ fib′ m (n + m)
-- fib = fib′ 0 1

fib : ∀{Γ} → Tm Γ Str
fib = box
      fib′ ∙ zero ∙ (suc zero)
      wth ⟨⟩
  where
    fib′ = fix-fib ∙ (ƛ ƛ ƛ fold ⟨ var (suc zero) , var (suc (suc zero)) ⊛ next var zero ⊛ next (var (suc zero) ⊹ (var zero)) ⟩)

-- A stream containing the same Fibonacci sequence, but defined with a stream
-- function (this term cannot be defined in Agda without sized types):

-- fib-ho = 0 ∷ 1 ∷ plus fib (gtl fib)

fib-ho : ∀{Γ} → Tm Γ Str
fib-ho = box fix-gStr ∙
         (ƛ (cons ∙ zero ∙ (next (cons ∙ suc zero) ⊛ (next next plus ⊛₂ next var zero ⊛₂ (next gtl ⊛ var zero)))))
         wth ⟨⟩

-- Paper folding sequence / Dragon sequence
-- http://en.wikipedia.org/wiki/Dragon_curve#.5BUn.5DFolding_the_Dragon

-- toggle = 1 ∷ 0 ∷ toggle
-- paperfolds = interleave toggle paperfolds

toggle : ∀{Γ} → Tm Γ gStr
toggle = fix-gStr ∙ (ƛ cons ∙ (suc zero) ∙ next (cons ∙ zero ∙ var zero))

paperfolds : ∀{Γ} → Tm Γ Str
paperfolds = box
             fix-gStr ∙ (ƛ interleave ∙ toggle ∙ var zero)
             wth ⟨⟩

-- The μ-types are unique fixed points, so carry both final coalgebra ans
-- initial algebra structure. The function foldr is a witness of the latter.

-- foldr : ((ℕ × ▸ A) → A) → gStr → A
-- foldr f (x ∷ xs) = f ⟨ x , foldr f xs ⟩

-- Since we do not have polymorphism, we need to define one for every type A we
-- need.

foldr-gStr : ∀{Γ} → Tm Γ (((ℕₜ ×ₜ ▸ gStr) →ₜ gStr) →ₜ gStr →ₜ gStr)
foldr-gStr = fix-foldr-gStr ∙
  (ƛ ƛ ƛ var (suc zero) ∙ ⟨ ghd ∙ var zero , var (suc (suc zero)) ⊛ next var (suc zero) ⊛ (gtl ∙ var zero) ⟩)

-- Using foldr we can define map

-- map : (ℕ → ℕ) → gStr → gStr
-- map f (x ∷ xs) = f x ∷ map f xs

map : ∀{Γ} → Tm Γ ((ℕₜ →ₜ ℕₜ) →ₜ gStr →ₜ gStr)
map = ƛ foldr-gStr ∙ (ƛ fold ⟨ var (suc zero) ∙ π₁ (var zero) , π₂ (var zero) ⟩)

-- Picking out every second element of a stream

-- every2nd (x ∷ y ∷ xs) = x ∷ every2nd xs

-- This is an acausal function. Therefore it cannot be implemented without using box.

every2nd :  ∀{Γ} → Tm Γ (Str →ₜ gStr)
every2nd = fix-every2nd ∙ (ƛ ƛ cons ∙ (ghd ∙ unbox (var zero)) ∙ var (suc zero) ⊛
                          next box prev prev next gtl ⊛ (gtl ∙ unbox (var zero))
                            wth var zero wth var zero wth var zero)
                            
