

module Substitutions where

open import Library

open import Types

open import Terms

------------------------------------------------------------------------------
-- Adapted from http://www2.tcs.ifi.lmu.de/~abel/ParallelSubstitution.agda

-- A structurally ordered two-element set.  The index is used to impose an order
-- on the constructors and so pass termination checking in lift/subst.

data VarTm : ℕ → Set where
  `Var : VarTm 0 -- we are dealing with variables
  `Tm  : VarTm 1 -- we are dealing with terms

-- A set of variables or terms

VT : ∀ {i} → (vt : VarTm i) → Cxt → ClTy → Set
VT `Var Γ A = Var Γ A
VT `Tm  Γ A = Tm Γ A

-- Embedding of VT into Tm

vt2tm : ∀ {Γ A} → ∀ {i} vt → VT {i} vt Γ A → Tm Γ A
vt2tm `Var x = var x
vt2tm `Tm t  = t

-- Either a renaming or a substitution

RenSub : ∀ {i} → VarTm i → Cxt → Cxt → Set
RenSub vt Γ Δ = ∀ {A} → Var Γ A → VT vt Δ A

mutual

  -- lifting a substitution

  lifts : ∀ {i vt Γ Δ A} → RenSub {i} vt Γ Δ → RenSub vt (A ∷ Γ) (A ∷ Δ)
  lifts {vt = `Var} σ zero    = zero
  lifts {vt = `Var} σ (suc x) = suc (σ x)
  lifts {vt = `Tm}  σ zero    = var zero
  lifts {vt = `Tm}  σ (suc x) = subst {vt = `Var} suc (σ x)

  -- performing a substitution

  subst : ∀ {i vt Γ Δ A} → RenSub {i} vt Γ Δ → Tm Γ A → Tm Δ A
  subst σ (var x) = vt2tm _ (σ x)
  subst σ (abort t) = abort (subst σ t)
  subst σ ⟨⟩ = ⟨⟩
  subst σ zero = zero
  subst σ (suc t) = suc (subst σ t)
  subst σ (t ⊹ t₁) = subst σ t ⊹ subst σ t₁
  subst σ (ƛ t) = ƛ (subst (lifts σ) t)
  subst σ (t ∙ t₁) = subst σ t ∙ subst σ t₁
  subst σ ⟨ t , t₁ ⟩ = ⟨ subst σ t , subst σ t₁ ⟩
  subst σ (π₁ t) = π₁ (subst σ t)
  subst σ (π₂ t) = π₂ (subst σ t)
  subst σ (in₁ t) = in₁ (subst σ t)
  subst σ (in₂ t) = in₂ (subst σ t)
  subst σ (cse t of t₁ , t₂) = cse subst σ t of subst (lifts σ) t₁ , subst (lifts σ) t₂
  subst σ (next t) = next (subst σ t)
  subst σ (prev t wth u) = prev t wth subst σ u
  subst σ (box t wth u) = box t wth subst σ u
  subst σ (unbox t) = unbox (subst σ t)
  subst σ (box+ t wth u) = box+ t wth subst σ u
  subst σ (t ⊛ t₁) = subst σ t ⊛ subst σ t₁
  subst σ (fold t) = fold (subst σ t)
  subst σ (unfold t) = unfold (subst σ t)

-- Identity substitution

ids : ∀ {i vt Γ} → RenSub {i} vt Γ Γ
ids {vt = `Var} x = x
ids {vt = `Tm} x = var x

-- Term substitution

Subst : Cxt → Cxt → Set
Subst Γ Δ = {A : ClTy} → Var Γ A → Tm Δ A

-- Extending a substitution

_∷s_ : ∀ {Γ Δ A} → Tm Γ A → Subst Δ Γ → Subst (A ∷ Δ) Γ
(t ∷s σ) zero = t
(t ∷s σ) (suc x) = σ x

-- Substitution for 0th variable

sgs : ∀ {Γ A} → Tm Γ A → Subst (A ∷ Γ) Γ
sgs t = t ∷s ids

-- Substituting for the 0th variable

subst0 : ∀ {Γ A B} → Tm Γ A → Tm (A ∷ Γ) B → Tm Γ B
subst0 u = subst (sgs u)

_[_/x] : ∀ {Γ A B} → Tm (A ∷ Γ) B → Tm Γ A → Tm Γ B
t [ u /x] = subst0 u t

-- Renamings

Ren : Cxt → Cxt → Set
Ren = RenSub `Var

_≤_ : Cxt → Cxt → Set
Γ ≤ Δ = Ren Δ Γ

rename : ∀{Γ Δ A} (η : Γ ≤ Δ) (x : Tm Δ A) → Tm Γ A
rename = subst

-- Weakening renaming

weak : ∀{Γ A} → (A ∷ Γ) ≤ Γ
weak = suc


