

module Terms where

open import Library

open import Types

import Relation.Binary.PropositionalEquality as PropEq

infix 4 ƛ_ ƛᵥ_
infixl 5 _∙_ _⊹_
infixl 6 _⊛_ _⊛₂_
infix 7 next_ nextᵥ_ 
infix 20 box_wth_ boxᵥ_wth_ prev_wth_ box′_

-- Variable contexts, de Bruijn representation

Cxt = List ClTy

-- Well-typed term variables

data Var : Cxt → ClTy → Set where
  zero : ∀{Γ A}                    → Var (A ∷ Γ) A
  suc  : ∀{Γ A B} → (x : Var Γ A) → Var (B ∷ Γ) A

-- Well-typed terms

data Tm (Γ : Cxt) : (A : ClTy) → Set where
  var : ∀{A} → (x : Var Γ A) → Tm Γ A
  abort : ∀{A} → (t : Tm Γ 0ₜ) → Tm Γ A
  ⟨⟩ : Tm Γ 1ₜ
  zero : Tm Γ ℕₜ
  suc : (t : Tm Γ ℕₜ) → Tm Γ ℕₜ
  _⊹_ : (t₁ : Tm Γ ℕₜ) (t₂ : Tm Γ ℕₜ) → Tm Γ ℕₜ
  ƛ_ : ∀{A B} (t : Tm (A ∷ Γ) B) → Tm Γ (A →ₜ B)
  _∙_ : ∀{A B} (t₁ : Tm Γ (A →ₜ B)) (t₂ : Tm Γ A) → Tm Γ B
  ⟨_,_⟩ : ∀{A B} (t₁ : Tm Γ A) (t₂ : Tm Γ B) → Tm Γ (A ×ₜ B)
  π₁ : ∀{A₁ A₂} (t : Tm Γ (A₁ ×ₜ A₂)) → Tm Γ A₁
  π₂ : ∀{A₁ A₂} (t : Tm Γ (A₁ ×ₜ A₂)) → Tm Γ A₂
  in₁ : ∀{A₁ A₂} (t : Tm Γ A₁) → Tm Γ (A₁ +ₜ A₂)
  in₂ : ∀{A₁ A₂} (t : Tm Γ A₂) → Tm Γ (A₁ +ₜ A₂)
  cse_of_,_ : ∀{A B₁ B₂} (t : Tm Γ (B₁ +ₜ B₂)) (u₁ : Tm (B₁ ∷ Γ) A) (u₂ : Tm (B₂ ∷ Γ) A) → Tm Γ A
  next_ : ∀{A} (t : Tm Γ A) → Tm Γ (▸ₜ A)
  prev_wth_ : ∀{A B} (t : Tm ((constant B) ∷ []) (▸ₜ A)) (u : Tm Γ (constant B)) → Tm Γ A
  box_wth_ : ∀{A B} (t : Tm ((constant B) ∷ []) A) (u : Tm Γ (constant B)) → Tm Γ (■ₜ A)
  unbox : ∀{A} (t : Tm Γ (■ₜ A)) → Tm Γ A
  box+_wth_ : ∀{B A₁ A₂} (t : Tm ((constant B) ∷ []) (A₁ +ₜ A₂)) (u : Tm Γ (constant B)) → Tm Γ (■ₜ A₁ +ₜ ■ₜ A₂)
  _⊛_ : ∀{A B} (t₁ : Tm Γ (▸ₜ_ {Θ′ = []} (A →ₜ B))) (t₂ : Tm Γ (▸ₜ A)) → Tm Γ (▸ₜ B)
  fold : ∀{A} (t : Tm Γ (A [ (μₜ A) /α])) → Tm Γ (μₜ A)
  unfold : ∀{A} (t : Tm Γ (μₜ A)) → Tm Γ (A [ (μₜ A) /α])

-- Closed terms

ClTm : ClTy → Set
ClTm = Tm []

-- Well-typed values

data Val : (A : ClTy) → Set where
  ⟨⟩ᵥ : Val 1ₜ
  zeroᵥ : Val ℕₜ
  sucᵥ : (v : Val ℕₜ) → Val ℕₜ
  ƛᵥ_ : ∀{A B} (t : Tm (A ∷ []) B) → Val (A →ₜ B)
  ⟨_,_⟩ᵥ : ∀{A B} (t₁ : Tm [] A) (t₂ : Tm [] B) → Val (A ×ₜ B)
  in₁ᵥ : ∀{A₁ A₂} (t : Tm [] A₁) → Val (A₁ +ₜ A₂)
  in₂ᵥ : ∀{A₁ A₂} (t : Tm [] A₂) → Val (A₁ +ₜ A₂)
  foldᵥ : ∀{A} (t : Tm [] (A [ (μₜ A) /α])) → Val (μₜ A)
  nextᵥ_ : ∀{A} (t : Tm [] A) → Val (▸ₜ A)
  boxᵥ_wth_ : ∀{A B} (t : Tm ((constant B) ∷ []) A) (u : Tm [] (constant B)) → Val (■ₜ A)

-- Val Γ A ⊆ Tm Γ A

value : ∀{A} → Val A → ClTm A
value ⟨⟩ᵥ = ⟨⟩
value zeroᵥ = zero
value (sucᵥ v) = suc (value v)
value (ƛᵥ t) = ƛ t
value ⟨ t₁ , t₂ ⟩ᵥ = ⟨ t₁ , t₂ ⟩
value (in₁ᵥ t) = in₁ t 
value (in₂ᵥ t) = in₂ t
value (foldᵥ t) = fold t
value (nextᵥ t) = next t
value (boxᵥ t wth u) = box t wth u


-- It is an injection

suc-inj : {t t′ : ClTm ℕₜ} → Tm.suc t ≡ suc t′ → t ≡ t′
suc-inj refl = refl

value-inj : ∀{A} {v v′ : Val A} → value v ≡ value v′ → v ≡ v′
value-inj {v = ⟨⟩ᵥ} {⟨⟩ᵥ} refl = refl
value-inj {v = zeroᵥ} {zeroᵥ} refl = refl
value-inj {v = zeroᵥ} {sucᵥ v′} ()
value-inj {v = sucᵥ v} {zeroᵥ} ()
value-inj {v = sucᵥ v} {sucᵥ v′} [v] = PropEq.cong sucᵥ (value-inj (suc-inj [v]))
value-inj {v = ƛᵥ t} {ƛᵥ .t} refl = refl
value-inj {v = ⟨ t₁ , t₂ ⟩ᵥ} {⟨ .t₁ , .t₂ ⟩ᵥ} refl = refl
value-inj {v = in₁ᵥ t} {in₁ᵥ .t} refl = refl
value-inj {v = in₁ᵥ t} {in₂ᵥ t₁} ()
value-inj {v = in₂ᵥ t} {in₁ᵥ t₁} ()
value-inj {v = in₂ᵥ t} {in₂ᵥ .t} refl = refl
value-inj {v = foldᵥ t} {foldᵥ .t} refl = refl
value-inj {v = nextᵥ t} {nextᵥ .t} refl = refl
value-inj {v = boxᵥ t wth u} {boxᵥ .t wth .u} refl = refl


-- Weakening of variable contexts

weakeningVar : ∀{Γ Δ A} → Var Γ A → Var (Γ ++L Δ) A
weakeningVar zero = zero
weakeningVar (suc x) = suc (weakeningVar x)

weakening : ∀{Γ Δ A} → Tm Γ A → Tm (Γ ++L Δ) A
weakening (var x) = var (weakeningVar x)
weakening (abort t) = abort (weakening t)
weakening ⟨⟩ = ⟨⟩
weakening zero = zero
weakening (suc t) = suc (weakening t)
weakening (t₁ ⊹ t₂) = (weakening t₁) ⊹ (weakening t₂)
weakening (ƛ t) = ƛ (weakening t)
weakening (t₁ ∙ t₂) = (weakening t₁) ∙ (weakening t₂)
weakening ⟨ t₁ , t₂ ⟩ = ⟨ (weakening t₁) , (weakening t₂) ⟩
weakening (π₁ t) = π₁ (weakening t)
weakening (π₂ t) = π₂ (weakening t)
weakening (in₁ t) = in₁ (weakening t)
weakening (in₂ t) = in₂ (weakening t)
weakening (cse t of u₁ , u₂) = cse weakening t of weakening u₁ , weakening u₂
weakening (next t) = next (weakening t)
weakening (prev_wth_ {B = B} t u) = prev_wth_ {B = B} t (weakening u)
weakening (box t wth u) = box t wth weakening u
weakening (unbox t) = unbox (weakening t)
weakening (box+ t wth u) = box+ t wth weakening u
weakening (t₁ ⊛ t₂) = (weakening t₁) ⊛ (weakening t₂)
weakening (fold t) = fold (weakening t)
weakening (unfold t) = unfold (weakening t)

-- A special case of the prev, box, box+ rule for closed terms

prev′_ : ∀{A} → ClTm (▸ A) → ClTm A
prev′ t = prev weakening t wth ⟨⟩

box′_ : ∀{A} → ClTm A → ClTm (■ₜ A)
box′ t = box weakening t wth ⟨⟩

box+′_ : ∀{A₁ A₂} → ClTm (A₁ +ₜ A₂) → ClTm (■ₜ A₁ +ₜ ■ₜ A₂)
box+′ t = box+ weakening t wth ⟨⟩

-- A version of ⊛ for two ▸'s.

_⊛₂_ : ∀{Γ A B} → Tm Γ (▸ ▸ (A →ₜ B)) → Tm Γ (▸ ▸ A) → Tm Γ (▸ ▸ B)
f ⊛₂ t = (next (ƛ ƛ var (suc zero) ⊛ var zero)) ⊛ f ⊛ t

-- Consistency. This is immediate from the soundness of the denotational
-- semantics (see paper).

postulate
  consistency : ClTm 0ₜ → ⊥
