

module Types where

open import Library

import Relation.Binary.PropositionalEquality as PropEq

import Data.Nat as N
import Data.Nat.Properties as N
import Data.Nat.Properties.Simple as N

import Data.Vec as Vec
import Data.Vec.Equality as Vec

import Data.Fin as Fin


infix 10 ▸ₜ_ ▸_
infix 10 ■ₜ_
infixr 8  _×ₜ_
infixr 7 _+ₜ_

infixr 6  _→ₜ_
infix 4  μₜ_

infix 2 _≅_ _≅V_

-----------------
--- Types -------

-- We introduce a datatype "Guardedness", which will be used for kinding the
-- type variables. The semantics of ⛨ will be "this variable is guarded by a ▸",
-- while ¬⛨ will mean "this variable might not be guarded".

open import Data.Bool using () renaming
  (
    Bool  to Guardedness;
    true  to ⛨;
    false to ¬⛨;
    _∧_   to _∧G_
  )
  public

-- An order on guardedness (standard ordering of bools)

_≤G_ : Guardedness → Guardedness → Set
g₁ ≤G g₂ = g₁ ≡ g₁ ∧G g₂

-- A type context is a list of type variables with their kind. We use de Bruijn
-- representation.

TyCxt = Vec Guardedness

-- Equality of TyCxt

open module TyCxtEq = Vec.PropositionalEquality {A = Guardedness}
  using ()
  renaming (_≈_ to _≈G_)
  
-- Lifting the guardedness ordering to type contexts

_≼_ : ∀{n} → TyCxt n → TyCxt n → Set
_≼_ = ∼V _≤G_

⛨s : ∀{n} → TyCxt n
⛨s = Vec.replicate ⛨

¬⛨s : ∀{n} → TyCxt n
¬⛨s = Vec.replicate ¬⛨

-- Well-kinded type variables

data TyVar : ∀{n} → TyCxt n → Set where
  zero : ∀{n} {Θ : TyCxt n}
    → TyVar (¬⛨ ∷ Θ)
  suc  : ∀{n} {g : Guardedness} {Θ : TyCxt n} (α : TyVar Θ)
    → TyVar (g ∷ Θ)

tyVar→Fin : ∀{n} {Θ : TyCxt n} → TyVar Θ → Fin n
tyVar→Fin zero = Fin.zero
tyVar→Fin (suc α) = Fin.suc (tyVar→Fin α)

tyVar→ℕ : ∀{n} {Θ : TyCxt n} → TyVar Θ → ℕ
tyVar→ℕ = Ftoℕ ∘ tyVar→Fin

-- Variable congruence

data _≅V_ : ∀{n m} {Θ : TyCxt n} {Θ′ : TyCxt m} (α : TyVar Θ) (α′ : TyVar Θ′) → Set where

  zero : ∀{n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
         ([Θ] : Θ ≈G Θ′) → zero {n} {Θ} ≅V zero {m} {Θ′}

  suc  : ∀{n m g} {Θ : TyCxt n} {Θ′ : TyCxt m} {α : TyVar Θ} {α′ : TyVar Θ′}
         ([α] : α ≅V α′) → suc {n} {g} {Θ} α ≅V suc {m} {g} {Θ′} α′

-- It is an equivalence

Vrefl : ∀ {n} {Θ : TyCxt n} {α : TyVar Θ} → α ≅V α
Vrefl {Θ = .¬⛨ ∷ Θ} {zero} = zero (TyCxtEq.refl Θ)
Vrefl {Θ = g ∷ Θ} {suc α} = suc Vrefl

Vsym : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {α : TyVar Θ} {α′ : TyVar Θ′}
       → α ≅V α′ → α′ ≅V α
Vsym (zero [Θ]) = zero (TyCxtEq.sym [Θ])
Vsym (suc [α]) = suc (Vsym [α])

Vtrans : ∀ {n m k} {Θ₁ : TyCxt n} {Θ₂ : TyCxt m} {Θ₃ : TyCxt k}
           {α : TyVar Θ₁} {β : TyVar Θ₂} {γ : TyVar Θ₃}
           → α ≅V β → β ≅V γ → α ≅V γ
Vtrans (zero [Θ]) (zero [Θ]₁) = zero (TyCxtEq.trans [Θ] [Θ]₁)
Vtrans (suc eq) (suc eq′) = suc (Vtrans eq eq′)

≅V-to-≈G : ∀ {n} {Θ Θ′ : TyCxt n} {A : TyVar Θ} {B : TyVar Θ′} → A ≅V B → Θ ≈G Θ′
≅V-to-≈G (zero [Θ]) = refl Vec.Equality.∷-cong [Θ]
≅V-to-≈G (suc eq) = refl Vec.Equality.∷-cong ≅V-to-≈G eq

≅V-to-≡ : ∀ {n} {Θ : TyCxt n} {α α′ : TyVar Θ} → α ≅V α′ → α ≡ α′
≅V-to-≡ (zero [Θ]) = refl
≅V-to-≡ (suc eq) = PropEq.cong suc (≅V-to-≡ eq)

-- Well-kinded types

data Ty {n : ℕ} (Θ : TyCxt n) : Set where  
  Varₜ         : (α : TyVar Θ) → Ty Θ
  0ₜ           : Ty Θ
  1ₜ           : Ty Θ
  ℕₜ           : Ty Θ
  _×ₜ_         : (A₁ : Ty Θ) (A₂ : Ty Θ) → Ty Θ
  _→ₜ_         : (A₁ : Ty Θ) (A₂ : Ty Θ) → Ty Θ
  _+ₜ_         : (A₁ : Ty Θ) (A₂ : Ty Θ) → Ty Θ
  ▸ₜ_           : {Θ′ : TyCxt n} (A : Ty Θ′) → Ty Θ
  μₜ_          : (A : Ty (⛨ ∷ Θ)) → Ty Θ
  ■ₜ_           : (A : Ty []) → Ty Θ

-- Well-kinded constant types (constant: all ▸'s are under ■'s)

data CstTy {n : ℕ} (Θ : TyCxt n) : Set where
  Varₜ         : (α : TyVar Θ) → CstTy Θ
  0ₜ           : CstTy Θ
  1ₜ           : CstTy Θ
  ℕₜ           : CstTy Θ
  _×ₜ_         : (A₁ : CstTy Θ) (A₂ : CstTy Θ) → CstTy Θ
  _→ₜ_         : (A₁ : CstTy Θ) (A₂ : CstTy Θ) → CstTy Θ
  _+ₜ_         : (A₁ : CstTy Θ) (A₂ : CstTy Θ) → CstTy Θ
  μₜ_          : (A : CstTy (⛨ ∷ Θ)) → CstTy Θ  -- should only happen when μ won't bind anything in A!
  ■ₜ_          : (A : Ty []) → CstTy Θ
  

-- Constant Types ⊆ Types

constant : ∀{n} {Θ : TyCxt n} → CstTy Θ → Ty Θ
constant (Varₜ α) = Varₜ α
constant 0ₜ = 0ₜ
constant 1ₜ = 1ₜ
constant ℕₜ = ℕₜ
constant (A ×ₜ A₁) = constant A ×ₜ constant A₁
constant (A →ₜ A₁) = constant A →ₜ constant A₁
constant (A +ₜ A₁) = constant A +ₜ constant A₁
constant (μₜ A) = μₜ constant A
constant (■ₜ A) = ■ₜ A

-- Closed guarded recursive types

ClTy = Ty []

-- Agda can't seem to find inhabitants of TyCxt zero automatically, and since we
-- are mostly going to work with closed types, it gets a bit annoying to write
-- ▸ₜ_ {Θ′ = []} A.

▸_ : ClTy → ClTy
▸ A = ▸ₜ A

-- Weakening of guardedness

weakGTyVar : ∀ {n} {Θ Θ′ : TyCxt n} → TyVar Θ → Θ′ ≼ Θ → TyVar Θ′
weakGTyVar zero (_∷_ {x = ⛨} () le)
weakGTyVar zero (_∷_ {x = ¬⛨} x∼y le) = zero
weakGTyVar {Θ′ = x ∷ Θ′} (suc α) (x∼y ∷ le) = suc (weakGTyVar α le)

weakGTy : ∀{n} {Θ Θ′ : TyCxt n} → Ty Θ → Θ′ ≼ Θ → Ty Θ′
weakGTy (Varₜ α) le = Varₜ (weakGTyVar α le)
weakGTy 0ₜ le = 0ₜ
weakGTy 1ₜ le = 1ₜ
weakGTy ℕₜ le = ℕₜ
weakGTy (A₁ ×ₜ A₂) le = weakGTy A₁ le ×ₜ weakGTy A₂ le
weakGTy (A₁ →ₜ A₂) le = weakGTy A₁ le →ₜ weakGTy A₂ le
weakGTy (A₁ +ₜ A₂) le = weakGTy A₁ le +ₜ weakGTy A₂ le
weakGTy (▸ₜ A) le = ▸ₜ A
weakGTy (μₜ A) le = μₜ weakGTy A (PropEq.refl ∷ le)
weakGTy (■ₜ A) le = ■ₜ A

-- Type congruence (extension of variable congruence)

data _≅_ : ∀{n₁ n₂} {Θ₁ : TyCxt n₁} {Θ₂ : TyCxt n₂} (A : Ty Θ₁) (B : Ty Θ₂) → Set where

  Varₜ     : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {α : TyVar Θ} {α′ : TyVar Θ′}
            ([α] : α ≅V α′) → Varₜ α ≅ Varₜ α′

  0ₜ       : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               ([Θ] : Θ ≈G Θ′) → _≅_ {n} {m} {Θ} {Θ′} 0ₜ 0ₜ

  1ₜ       : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               ([Θ] : Θ ≈G Θ′) → _≅_ {n} {m} {Θ} {Θ′} 1ₜ 1ₜ

  ℕₜ       : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               ([Θ] : Θ ≈G Θ′) → _≅_ {n} {m} {Θ} {Θ′} ℕₜ ℕₜ

  _×ₜ_     : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               {A₁ : Ty Θ} {A₂ : Ty Θ′} {B₁ : Ty Θ} {B₂ : Ty Θ′}
               ([A] : A₁ ≅ A₂) ([B] : B₁ ≅ B₂) → (A₁ ×ₜ B₁) ≅ (A₂ ×ₜ B₂)

  _→ₜ_     : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               {A₁ : Ty Θ} {A₂ : Ty Θ′} {B₁ : Ty Θ} {B₂ : Ty Θ′}
               ([A] : A₁ ≅ A₂) ([B] : B₁ ≅ B₂) → (A₁ →ₜ B₁) ≅ (A₂ →ₜ B₂)

  _+ₜ_     : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m}
               {A₁ : Ty Θ} {A₂ : Ty Θ′} {B₁ : Ty Θ} {B₂ : Ty Θ′}
               ([A] : A₁ ≅ A₂) ([B] : B₁ ≅ B₂) → (A₁ +ₜ B₁) ≅ (A₂ +ₜ B₂)


  ▸ₜ_        : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {A : Ty Θ} {A′ : Ty Θ′}
               {Δ : TyCxt n} {Δ′ : TyCxt m} ([A] : A ≅ A′) ([Δ] : Δ ≈G Δ′)
               → _≅_ {n} {m} {Δ} {Δ′} (▸ₜ A) (▸ₜ A′)

  μₜ_       : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {A : Ty (⛨ ∷ Θ)} {A′ : Ty (⛨ ∷ Θ′)}
               ([A] : A ≅ A′) → μₜ A ≅ μₜ A′

  ■ₜ_       : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {A : Ty []} {A′ : Ty []}
               ([Θ] : Θ ≈G Θ′) ([A] : A ≅ A′) → _≅_ {Θ₁ = Θ} {Θ₂ = Θ′} (■ₜ A) (■ₜ A′)


-- It is an equivalence

Trefl : ∀ {n} {Θ : TyCxt n} {A : Ty Θ} → A ≅ A
Trefl {A = Varₜ α} = Varₜ Vrefl
Trefl {Θ = Θ} {A = 0ₜ} = 0ₜ (TyCxtEq.refl Θ)
Trefl {Θ = Θ} {A = 1ₜ} = 1ₜ (TyCxtEq.refl Θ)
Trefl {Θ = Θ} {A = ℕₜ} = ℕₜ (TyCxtEq.refl Θ)
Trefl {A = A ×ₜ A₁} = Trefl ×ₜ Trefl
Trefl {A = A →ₜ A₁} = Trefl →ₜ Trefl
Trefl {A = A +ₜ A₁} = Trefl +ₜ Trefl
Trefl {Θ = Θ} {A = ▸ₜ A} = (▸ₜ Trefl) (TyCxtEq.refl Θ)
Trefl {n} {Θ} {A = μₜ A} = μₜ_ {m = n} {Θ′ = Θ} Trefl
Trefl {Θ = Θ} {A = ■ₜ A} = (■ₜ (TyCxtEq.refl Θ)) Trefl

Tsym : ∀ {n m} {Θ : TyCxt n} {Θ′ : TyCxt m} {A : Ty Θ} {A′ : Ty Θ′}
         → A ≅ A′ → A′ ≅ A
Tsym (Varₜ [α]) = Varₜ (Vsym [α])
Tsym (0ₜ [Θ]) = 0ₜ (TyCxtEq.sym [Θ])
Tsym (1ₜ [Θ]) = 1ₜ (TyCxtEq.sym [Θ])
Tsym (ℕₜ [Θ]) = ℕₜ (TyCxtEq.sym [Θ])
Tsym ([A] ×ₜ [A]₁) = Tsym [A] ×ₜ Tsym [A]₁
Tsym ([A] →ₜ [A]₁) = Tsym [A] →ₜ Tsym [A]₁
Tsym ([A] +ₜ [A]₁) = Tsym [A] +ₜ Tsym [A]₁
Tsym (▸ₜ_ [A] [Δ]) = (▸ₜ (Tsym [A])) (TyCxtEq.sym [Δ])
Tsym (μₜ [A]) = μₜ (Tsym [A])
Tsym (■ₜ_ [Θ] [A]) = (■ₜ (TyCxtEq.sym [Θ])) (Tsym [A])

Ttrans : ∀ {n m k} {Θ₁ : TyCxt n} {Θ₂ : TyCxt m} {Θ₃ : TyCxt k}
           {A : Ty Θ₁} {B : Ty Θ₂} {C : Ty Θ₃}
           → A ≅ B → B ≅ C → A ≅ C
Ttrans (Varₜ [α]) (Varₜ [α]₁) = Varₜ (Vtrans [α] [α]₁)
Ttrans (0ₜ [Θ]) (0ₜ [Θ]₁) = 0ₜ (TyCxtEq.trans [Θ] [Θ]₁)
Ttrans (1ₜ [Θ]) (1ₜ [Θ]₁) = 1ₜ (TyCxtEq.trans [Θ] [Θ]₁)
Ttrans (ℕₜ [Θ]) (ℕₜ [Θ]₁) = ℕₜ (TyCxtEq.trans [Θ] [Θ]₁)
Ttrans (eq ×ₜ eq₁) (eq′ ×ₜ eq′₁) = Ttrans eq eq′ ×ₜ Ttrans eq₁ eq′₁
Ttrans (eq →ₜ eq₁) (eq′ →ₜ eq′₁) = Ttrans eq eq′ →ₜ Ttrans eq₁ eq′₁
Ttrans (eq +ₜ eq₁) (eq′ +ₜ eq′₁) = Ttrans eq eq′ +ₜ Ttrans eq₁ eq′₁
Ttrans (▸ₜ_ eq [Δ]) (▸ₜ_ eq′ [Δ]₁) = (▸ₜ (Ttrans eq eq′)) (TyCxtEq.trans [Δ] [Δ]₁)
Ttrans (μₜ eq) (μₜ eq′) = μₜ (Ttrans eq eq′)
Ttrans (■ₜ_ [Θ] eq) (■ₜ_ [Θ′] eq′) = (■ₜ (TyCxtEq.trans [Θ] [Θ′])) (Ttrans eq eq′)

≅-to-≈G : ∀ {n} {Θ Θ′ : TyCxt n} {A : Ty Θ} {B : Ty Θ′} → A ≅ B → Θ ≈G Θ′
≅-to-≈G (Varₜ [α]) = ≅V-to-≈G [α]
≅-to-≈G (0ₜ [Θ]) = [Θ]
≅-to-≈G (1ₜ [Θ]) = [Θ]
≅-to-≈G (ℕₜ [Θ]) = [Θ]
≅-to-≈G (eq ×ₜ eq₁) = ≅-to-≈G eq₁
≅-to-≈G (eq →ₜ eq₁) = ≅-to-≈G eq₁
≅-to-≈G (eq +ₜ eq₁) = ≅-to-≈G eq₁
≅-to-≈G (▸ₜ_ eq [Δ]) = [Δ]
≅-to-≈G (μₜ eq) with ≅-to-≈G eq
≅-to-≈G (μₜ eq) | refl Vec.Equality.∷-cong eq′ = eq′
≅-to-≈G {Θ = Θ} (■ₜ_ [Θ] eq) = [Θ]

≅-to-≡ : ∀ {n} {Θ : TyCxt n} {A B : Ty Θ} → A ≅ B → A ≡ B
≅-to-≡ (Varₜ [α]) = PropEq.cong Varₜ (≅V-to-≡ [α])
≅-to-≡ (0ₜ [Θ]) = refl
≅-to-≡ (1ₜ [Θ]) = refl
≅-to-≡ (ℕₜ [Θ]) = refl
≅-to-≡ (eq₁ ×ₜ eq₂) = PropEq.cong₂ _×ₜ_ (≅-to-≡ eq₁) (≅-to-≡ eq₂)
≅-to-≡ (eq₁ →ₜ eq₂) = PropEq.cong₂ _→ₜ_ (≅-to-≡ eq₁) (≅-to-≡ eq₂)
≅-to-≡ (eq₁ +ₜ eq₂) = PropEq.cong₂ _+ₜ_ (≅-to-≡ eq₁) (≅-to-≡ eq₂)
≅-to-≡ (▸ₜ_ eq [Δ]) with TyCxtEq.to-≡ (≅-to-≈G eq)
... | refl = PropEq.cong ▸ₜ_ (≅-to-≡ eq)
≅-to-≡ (μₜ eq) = PropEq.cong μₜ_ (≅-to-≡ eq)
≅-to-≡ (■ₜ_ [Θ] eq) = PropEq.cong ■ₜ_ (≅-to-≡ eq)

-- Weakening of type variable context length

weakeningTyVar : ∀ {n m} {Θ : TyCxt n} {Δ : TyCxt m} → TyVar Θ → TyVar (Θ ++V Δ)
weakeningTyVar zero = zero
weakeningTyVar (suc α) = suc (weakeningTyVar α)

weakeningTy : ∀{n m} {Θ : TyCxt n} {Δ : TyCxt m} → Ty Θ → Ty (Θ ++V Δ)
weakeningTy (Varₜ α) = Varₜ (weakeningTyVar α)
weakeningTy 0ₜ = 0ₜ
weakeningTy 1ₜ = 1ₜ
weakeningTy ℕₜ = ℕₜ
weakeningTy (A₁ ×ₜ A₂) = weakeningTy A₁ ×ₜ weakeningTy A₂
weakeningTy (A₁ →ₜ A₂) = weakeningTy A₁ →ₜ weakeningTy A₂
weakeningTy (A₁ +ₜ A₂) = weakeningTy A₁ +ₜ weakeningTy A₂
weakeningTy {Δ = Δ} (▸ₜ A) = ▸ₜ weakeningTy {Δ = Δ} A
weakeningTy (μₜ A) = μₜ weakeningTy A
weakeningTy (■ₜ A) = ■ₜ weakeningTy A

-- Properties about weakening

Θ++[]≈Θ : ∀{n} (Θ : TyCxt n) → Θ ++V [] ≈G Θ
Θ++[]≈Θ [] = TyCxtEq.[]-cong
Θ++[]≈Θ (A ∷ Θ) = PropEq.refl TyCxtEq.∷-cong Θ++[]≈Θ Θ 

wα[]≅α : ∀ {n} {Θ : TyCxt n} → (α : TyVar Θ) → weakeningTyVar {Δ = []} α ≅V α
wα[]≅α {suc n} {Θ = ¬⛨ ∷ Θ} zero = zero (Θ++[]≈Θ Θ)
wα[]≅α (suc α) = suc (wα[]≅α α)

wA[]≅A : ∀ {n} {Θ : TyCxt n} → (A : Ty Θ) → weakeningTy {Δ = []} A ≅ A
wA[]≅A (Varₜ α) = Varₜ (wα[]≅α α)
wA[]≅A {Θ = Θ} 0ₜ = 0ₜ (Θ++[]≈Θ Θ)
wA[]≅A {Θ = Θ} 1ₜ = 1ₜ (Θ++[]≈Θ Θ)
wA[]≅A {Θ = Θ} ℕₜ = ℕₜ (Θ++[]≈Θ Θ)
wA[]≅A (A ×ₜ A₁) = wA[]≅A A ×ₜ wA[]≅A A₁
wA[]≅A (A →ₜ A₁) = wA[]≅A A →ₜ wA[]≅A A₁
wA[]≅A (A +ₜ A₁) = wA[]≅A A +ₜ wA[]≅A A₁
wA[]≅A {Θ = Θ} (▸ₜ A) = (▸ₜ (wA[]≅A A)) (Θ++[]≈Θ Θ)
wA[]≅A (μₜ A) = μₜ wA[]≅A A
wA[]≅A {Θ = Θ} (■ₜ A) = (■ₜ Θ++[]≈Θ Θ) (wA[]≅A A)

----------------------------------------------
-- Parallel type substitutions of closed types

TySubst : ℕ → Set
TySubst = Vec ClTy

-- Reduce the kinding context of a variable, when possible

varTrans : ∀{n m} {Θ : TyCxt (m + n)} (α : TyVar Θ) → tyVar→ℕ α <ℕ m → TyVar (take m Θ)
varTrans {m = zero} α ()
varTrans {m = suc m} zero α<m = zero
varTrans {m = suc m} (suc α) α<m = suc (varTrans α (pred≤ℕ α<m))

-- Perform substitution

tySubst : ∀{m n} {Θ : TyCxt (m + n)} → Ty Θ → (σ : TySubst n) → Ty (take m Θ)
tySubst {m} {n} {Θ = Θ} (Varₜ α) σ with  m ≤ℕ? tyVar→ℕ α
... | yes p = weakeningTy (Vec.lookup (Fin.reduce≥ (tyVar→Fin α) p) σ)
... | no ¬p = Varₜ (varTrans α (≰-to-> ¬p))
tySubst 0ₜ σ = 0ₜ
tySubst 1ₜ σ = 1ₜ
tySubst ℕₜ σ = ℕₜ
tySubst (A₁ ×ₜ A₂) σ = tySubst A₁ σ ×ₜ tySubst A₂ σ
tySubst (A₁ →ₜ A₂) σ = tySubst A₁ σ →ₜ tySubst A₂ σ
tySubst (A₁ +ₜ A₂) σ = tySubst A₁ σ +ₜ tySubst A₂ σ
tySubst (▸ₜ A) σ = ▸ₜ tySubst A σ
tySubst (μₜ A) σ = μₜ tySubst A σ
tySubst (■ₜ A) σ = ■ₜ A

_[_/α] : ∀{g} → Ty (g ∷ []) → ClTy → ClTy
A [ B /α] = tySubst A (B ∷ [])

-- Local Variables:
-- eval: (agda-input-add-translations '(("gu" . "⛨")))
-- End:
